/* xballzup.c 
 *  X11 conversion by Nathan Laredo (laredo@gnu.org)
 */

#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <signal.h>
#include <sys/ipc.h>
#include <sys/shm.h>
#include <sys/time.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Intrinsic.h>
#include <X11/extensions/XShm.h>

#define SPHERE_SIZE	63U
#define SCREEN_WIDTH	480U
#define SCREEN_HEIGHT	480U

#define SCREEN_WIDTH_OVER_2	((float)(SCREEN_WIDTH / 2))
#define SCREEN_HEIGHT_OVER_2	((float)(SCREEN_HEIGHT / 2))
#define SPHERE_SIZE_FLOAT	((float)(SPHERE_SIZE))
#define SCREEN_WIDTH_FLOAT	((float)(SCREEN_WIDTH))

typedef unsigned char u8;
typedef unsigned short u16;
typedef unsigned int u32;

static const int g_width = SCREEN_WIDTH;
static const int g_height = SCREEN_HEIGHT;
static const char window_title[] = "Return of Ballzup";
static int g_depth;

static int buffer_depth[SCREEN_WIDTH * SCREEN_HEIGHT];
static int sphere_depth[SPHERE_SIZE * SPHERE_SIZE];

static Display *mydisplay;
static Window mywindow;
static XShmSegmentInfo shminfo;
static XImage *myximage;
static int myscreen, shmem_flag = 0, xerrorflag = 0;

static float frand(void)
{
	return (rand() % 6280) / 1000.0f;
}

static float frac(float x)
{
	return (x - floor(x));
}

static void sphere_generate(void)
{
	float x, y = -1;
	int sx, sy, *sd = sphere_depth;
	const float sgm_d = 2.0f / SPHERE_SIZE;

	for (sy = 0; sy < SPHERE_SIZE; sy++, y += sgm_d)
		for (sx = 0, x = -1; sx < SPHERE_SIZE; sx++, x += sgm_d)
			*sd++ = (1 - sqrt(1 - (x * x + y * y))) * 65535;
}

static void put_sprite_depth_scale(int *src_depth, u32 ssize, int *dst_depth,
				   u32 pitch, int depth, float scale)
{
	u32 x, y, ix, idx, new_size;
	int *dstd, *srcd;

	idx = (1.0f / scale) * 65536;
	new_size = ssize * scale;

	for (y = 0; y < new_size; y++) {
		dstd = dst_depth + y * pitch;
		srcd = src_depth + (int)floor(y * (1.0f / scale)) * ssize;
		ix = 0;

		for (x = 0; x < new_size; x++) {
			if (srcd[ix >> 16] > 0)
				if (srcd[ix >> 16] + depth < *dstd)
					*dstd = srcd[ix >> 16] + depth;
			dstd++;
			ix += idx;
		}
	}
}

static void draw_buffer()
{
	u32 *dst32 = (u32 *) myximage->data;
	u16 *dst16 = (u16 *) myximage->data;
	u8 *dst8 = (u8 *) myximage->data;
	int *src = buffer_depth;
	int i;

	switch (g_depth) {
	case 8:
		for (i = 0; i < SCREEN_HEIGHT * SCREEN_WIDTH; i++)
			*dst8++ = (u8)(*src++ * -0.00078 + 128);
		break;
	case 15:
	case 16:
		for (i = 0; i < SCREEN_HEIGHT * SCREEN_WIDTH; i++)
			*dst16++ = (((u8)(*src++ * -0.00078 + 128)) >> 2) << 5;
		break;
	case 24:
	case 32:
		for (i = 0; i < SCREEN_HEIGHT * SCREEN_WIDTH; i++)
			*dst32++ = ((u8)(*src++ * -0.00078 + 128)) << 8;
		break;
	default:
		fprintf(stderr, "don't know what to do with depth %d\n",
			g_depth);
		exit(1);
	}
	if (shmem_flag) {
		XShmPutImage(mydisplay, mywindow,
			     DefaultGC(mydisplay, myscreen),
			     myximage, 0, 0, 0, 0,
			     g_width, g_height, False);
	} else {
		XPutImage(mydisplay, mywindow,
			  DefaultGC(mydisplay, myscreen),
			  myximage, 0, 0, 0, 0, g_width, g_height);
	}
	XSync(mydisplay, False);	/* ensure above completes */
}

static void knot_func_0(float t, float *x, float *y, float *z, float *scale)
{
	*x = sin(t + frand());
	*y = cos(t + frand());
	*z = cos(t + frand());
	*scale = 0.3;
}

static void knot_func_1(float t, float *x, float *y, float *z, float *scale)
{
	*x = ((3 + sin(14 * t)) * cos(t)) * 0.4;
	*y = (cos(14 * t)) * 0.4;
	*z = ((3 + sin(14 * t)) * sin(t)) * 0.4;
	*scale = 0.4;
}

static void knot_func_crusoe(float t, float *x, float *y, float *z,
			     float *scale)
{
	*x = t * sin(2 * t) / (M_PI * 2);
	*y = t * 0.07;
	*z = t * cos(2 * t) / (M_PI * 2);
	*scale = 0.4f;
}
static void do_one_frame(void)
{
	static float t = 0.0, tt, blend;
	static struct timeval then, now;
	static int frame = 0;
	int *clearz = buffer_depth, kf;

	if (!frame)
		gettimeofday(&then, NULL);
	frame++;
	if (frame == 20) {
		gettimeofday(&now, NULL);
		now.tv_sec -= then.tv_sec;
		now.tv_usec -= then.tv_usec;
		if (now.tv_usec < 0)
			(now.tv_sec--, now.tv_usec += 1000000);
		fprintf(stderr, "\r%d frames in %ld.%06lds (%.4ffps)",
			frame, now.tv_sec, now.tv_usec,
			(float) frame * 1000000 /
			(float) (now.tv_sec * 1000000 + now.tv_usec));
		frame = 0;
		then = now;
	}
	while (clearz < buffer_depth + SCREEN_WIDTH * SCREEN_HEIGHT)
		*clearz++ = 800000;

	t += 0.015;
	kf = (int)((t * 0.5 + M_PI) / (2 * M_PI)) & 1;
	blend = cos(t * 0.5) * 0.5 + 0.5;
	srand(0);

	for (tt = 0.0; tt < 2 * M_PI; tt += M_PI / 180.0) {
		float x, y, z, x1, y1, z1, x2, y2, z2;
		float scale, scale1, scale2;
		float nx, ny, nz, ra, rb, rc;

		float biasedz, screen_scale, screen_x, screen_y;
		float sprite_size;
		u32 dboffset;

		knot_func_crusoe(tt, &x1, &y1, &z1, &scale1);
		if (kf)
			knot_func_0(tt, &x2, &y2, &z2, &scale2);
		else
			knot_func_1(tt, &x2, &y2, &z2, &scale2);


		x = x1 * (1 - blend) + x2 * blend;
		y = y1 * (1 - blend) + y2 * blend;
		z = z1 * (1 - blend) + z2 * blend;
		scale = scale1 * (1 - blend) + scale2 * blend;

		ra = sin(t * 1.2) * 5;
		rb = sin(t * 1.4) * 5;
		rc = sin(t * 1.6) * 5;

		nx = x * cos(rb) + z * -sin(rb);
		ny = y;
		nz = x * sin(rb) + z * cos(rb);

		x = nx;
		y = ny * cos(ra) + nz * -sin(ra);
		z = ny * sin(ra) + nz * cos(ra);

		nx = x * cos(rc) + y * -sin(rc);
		ny = x * sin(rc) + y * cos(rc);
		nz = z;

		x = nx; y = ny; z = nz;

		biasedz = z + 10;
		screen_scale = 1300;

		screen_x = (x / biasedz) * screen_scale +
				(SCREEN_WIDTH_OVER_2);
		screen_y = (y / biasedz) * screen_scale +
				(SCREEN_HEIGHT_OVER_2);
		sprite_size = (5.0f / biasedz) * scale;

		dboffset = (int)(screen_x - sprite_size * 0.5 * SPHERE_SIZE) +
			(int)(screen_y - sprite_size * 0.5 * SPHERE_SIZE) *
			SCREEN_WIDTH;

		put_sprite_depth_scale(sphere_depth, SPHERE_SIZE, buffer_depth
				       + dboffset, SCREEN_WIDTH, z * 65535,
				       sprite_size);
	}
	draw_buffer();
}

/* prevent a failure from causing an error exit */
static int handle_x_error(Display *mydisplay, XErrorEvent *event)
{
	char errortext[128];
	xerrorflag++;

	XGetErrorText(mydisplay, event->error_code, errortext, 128);
	fprintf(stderr, "xballzup: %s (%d/%d)\n", errortext,
		event->request_code, event->minor_code);
	return 0;
}

void init_display(int argc, char **argv)
{
	unsigned long myforeground, mybackground;
	XSizeHints myhint;

	if (!getenv("DISPLAY")) {
		fprintf(stderr, "DISPLAY not set.\n");
		exit(1);
	}
	if (!(mydisplay = XOpenDisplay(getenv("DISPLAY")))) {
		fprintf(stderr, "Unable to open display.\n");
		exit(1);
	}
	myscreen = DefaultScreen(mydisplay);
	mybackground = WhitePixel(mydisplay, myscreen);
	myforeground = BlackPixel(mydisplay, myscreen);
	g_depth = DefaultDepth(mydisplay, myscreen);
	myhint.width = g_width; 
	myhint.height = g_height;
	myhint.x = myhint.y = 0;
	myhint.max_width = g_width;
	myhint.max_height = g_height;
	myhint.min_width = g_width;
	myhint.min_height = g_height;
	myhint.flags = PSize | PMinSize | PMaxSize | PPosition;
	mywindow = XCreateSimpleWindow(mydisplay,
		DefaultRootWindow(mydisplay), myhint.x, myhint.y,
		myhint.width, myhint.height, 0, 0, mybackground);
	XSetStandardProperties(mydisplay, mywindow, window_title,
			       window_title, None, argv, argc, &myhint);
	XSelectInput(mydisplay, mywindow, ButtonPressMask | KeyPressMask);
	XMapRaised(mydisplay, mywindow);
	XSetErrorHandler(handle_x_error);
	XFlush(mydisplay);
}

int main(int argc, char **argv)
{
	init_display(argc, argv);

	/* allocate shared memory region for XShmPutImage */
	shminfo.shmid = shmget(IPC_PRIVATE, g_width * g_height * 4,
			       IPC_CREAT | 0777);
	if (shminfo.shmid < 0)
		exit(1);
	shminfo.shmaddr = (char *) shmat(shminfo.shmid, NULL, 0);
	shminfo.readOnly = False;
	shmctl(shminfo.shmid, IPC_RMID, 0);
	
	if (XShmQueryExtension(mydisplay)) {	/* use shared memory */
		XShmAttach(mydisplay, &shminfo);
		XSync(mydisplay, False);	/* ensure above completes */
		if (xerrorflag) {
			xerrorflag = 0;
			fprintf(stderr, "warning: XShm not available\n");
			goto shmerror;
		}
		myximage = XShmCreateImage(mydisplay, None, g_depth,
					   ZPixmap, shminfo.shmaddr, &shminfo,
					   g_width, g_height);
		if (!myximage)
			goto shmerror;
		shmem_flag++;
	} else {
	    shmerror:
		fprintf(stderr, "warning: Creating non-shared XImage\n");
		myximage = XCreateImage(mydisplay, None, g_depth,
					ZPixmap, 0, shminfo.shmaddr,
					g_width, g_height, 32, 0);
	}
	myximage->data = shminfo.shmaddr;
	
	sphere_generate();

	while (1) {
		do_one_frame();
	}
	exit(0);
}
