/* $Id: idtool.c,v 1.5 2006/02/02 23:39:55 nil Exp $
 * This is free software distributed under the GNU
 * General Public Licence Version 2
 * idtool.c -- Nathan Laredo <laredo@gnu.org>
 */

#ifdef WIN32

#include <windows.h>
/* define some C99 types for windows */
typedef unsigned __int32 uint32_t;
typedef unsigned __int16 uint16_t;
typedef unsigned __int64 uint64_t;
typedef unsigned __int8 uint8_t;

#else
#define _LARGEFILE64_SOURCE
#include <stdint.h>
#include <fcntl.h>
#include <string.h>
#include <unistd.h>
#include <sys/types.h>
#endif

#include <stdio.h>

static char messagestring[65536];
static uint32_t tm00info[32][4], tm8kinfo[32][4];
static uint32_t tm86info[32][4], tmc0info[32][4];
static uint32_t tmbainfo[32][4], allzero[4], mystery[4];

/* cpuid with abcd string order */
void do_cpuid(void *dest, uint32_t level, uint32_t tunnel)
{
#ifndef __GNUC__
	_asm {
		mov ebx, tunnel
		mov eax, level
		cpuid
		mov edi, dest
		mov [edi], eax
		mov [edi + 4], ebx
		mov [edi + 8], ecx
		mov [edi + 12], edx
	}
#else
	uint32_t *dst = dest;
	asm("push %%ebx\n\t"
	    "mov %%esi, %%ebx\n\t"
	    "cpuid\n\t"
	    "mov %%ebx, %%esi\n\t"
	    "pop %%ebx\n\t": "=a" (dst[0]), "=S" (dst[1]), "=c" (dst[2]), "=d" (dst[3])
	    : "a" (level), "r" (tunnel), "c" (tunnel));

#endif
}

/* Intel Application Note 485 - Order # 241618-027 - July 2004 */

static char *edx1_feat[32] = {
	"fpu","vme","de","pse","tsc","msr","pae","mce",
	"cx8","apic","res10","sep","mtrr","pge", "mca","cmov",
	"pat","pse36","psn","clfsh","res20","ds", "acpi","mmx",
	"fxsr","sse","sse2","ss","htt", "tm","ia64","pbe"
};

static char *ecx1_feat[32] = {
	"sse3","res1","res2","monitor","ds-cpl","vmx","res6","eist",
	"tm2","ssse3","cid","res11","res12","cx16","xtpr","res15",
	"res16","res17","res18","res19","res20","res21","res22","res23",
	"res24","res25","res26","res27","res28","res29","res30","raz31"
};

static char *cway[16] = {
	"L2 off", "direct mapped", "2-way", "res3", "4-way", "res5",
	"8-way", "res7", "16-way", "res9", "res10", "res11", "res12",
	"res13", "res14", "fully associative"
};

static char *ctype[4] = {
	"null", "data", "code", "unified"
};

#ifdef __GNUC__
/* MSVC lacks c99 initialization */
static char *cache_decode[256] = {
[0x01]="Instruction TLB: 4-KB Pages, 4-way set associative, 32 entries",
[0x02]="Instruction TLB: 4-MB Pages, fully associative, 2 entries",
[0x03]="Data TLB: 4-KB Pages, 4-way set associative, 64 entries",
[0x04]="Data TLB: 4-MB Pages, 4-way set associative, 8 entries",
[0x05]="Data TLB1: 4-MB Pages, 4-way set associative, 32 entries",
[0x06]="1st-level instruction cache: 8-KB, 4-way set associative, 32-byte line size",
[0x08]="1st-level instruction cache: 16-KB, 4-way set associative, 32-byte line size",
[0x0A]="1st-level data cache: 8-KB, 2-way set associative, 32-byte line size",
[0x0B]="Instruction TLB: 4-MB pages, 4-way set associative, 4 entries",
[0x0C]="1st-level data cache: 16-KB, 4-way set associative, 32-byte line size",
[0x22]="3rd-level cache: 512 KB, 4-way set associative, sectored cache, 64-byte line size",
[0x23]="3rd-level cache: 1-MB, 8-way set associative, sectored cache, 64-byte line size",
[0x25]="3rd-level cache: 2-MB, 8-way set associative, sectored cache, 64-byte line size",
[0x29]="3rd-level cache: 4-MB, 8-way set associative, sectored cache, 64-byte line size",
[0x2C]="1st-level data cache: 32-KB, 8-way set associative, 64-byte line size",
[0x30]="1st-level instruction cache: 32-KB, 8-way set associative, 64-byte line size",
[0x39]="2nd-level cache: 128-KB, 4-way set associative, sectored cache, 64-byte line size",
[0x3B]="2nd-level cache: 128-KB, 2-way set associative, sectored cache, 64-byte line size",
[0x3C]="2nd-level cache: 256-KB, 4-way set associative, sectored cache, 64-byte line size",
[0x40]="No 2nd-level cache or, if processor contains a valid 2nd-level cache, no 3rd-level cache",
[0x41]="2nd-level cache: 128-KB, 4-way set associative, 32-byte line size",
[0x42]="2nd-level cache: 256-KB, 4-way set associative, 32-byte line size",
[0x43]="2nd-level cache: 512-KB, 4-way set associative, 32-byte line size",
[0x44]="2nd-level cache: 1-MB, 4-way set associative, 32-byte line size",
[0x45]="2nd-level cache: 2-MB, 4-way set associative, 32-byte line size",
[0x49]="2nd-level cache: 4-MB, 16-way set associative, 64-byte line size",
[0x50]="Instruction TLB: 4-KB, 2-MB or 4-MB pages, fully associative, 64 entries",
[0x51]="Instruction TLB: 4-KB, 2-MB or 4-MB pages, fully associative, 128 entries",
[0x52]="Instruction TLB: 4-KB, 2-MB or 4-MB pages, fully associative, 256 entries",
[0x56]="Data TLB0: 4-MB pages, 4-way set associative, 16 entries",
[0x57]="Data TLB0: 4-KB pages, 4-way associative, 16 entries",
[0x5B]="Data TLB: 4-KB or 4-MB pages, fully associative, 64 entries",
[0x5C]="Data TLB: 4-KB or 4-MB pages, fully associative, 128 entries",
[0x5D]="Data TLB: 4-KB or 4-MB pages, fully associative, 256 entries",
[0x60]="1st-level data cache: 16-KB, 8-way set associative, sectored cache, 64-byte line size",
[0x66]="1st-level data cache: 8-KB, 4-way set associative, sectored cache, 64-byte line size",
[0x67]="1st-level data cache: 16-KB, 4-way set associative, sectored cache, 64-byte line size",
[0x68]="1st-level data cache: 32-KB, 4 way set associative, sectored cache, 64-byte line size",
[0x70]="Trace cache: 12K-uops, 8-way set associative",
[0x71]="Trace cache: 16K-uops, 8-way set associative",
[0x72]="Trace cache: 32K-uops, 8-way set associative",
[0x78]="2nd-level cache: 1-MB, 4-way set associative, 64-byte line size",
[0x79]="2nd-level cache: 128-KB, 8-way set associative, sectored cache, 64-byte line size",
[0x7A]="2nd-level cache: 256-KB, 8-way set associative, sectored cache, 64-byte line size",
[0x7B]="2nd-level cache: 512-KB, 8-way set associative, sectored cache, 64-byte line size",
[0x7C]="2nd-level cache: 1-MB, 8-way set associative, sectored cache, 64-byte line size",
[0x7D]="2nd-level cache: 2-MB, 8-way set associative, 64-byte line size",
[0x7F]="2nd-level cache: 512-KB, 2-way set associative, 64-byte line size",
[0x82]="2nd-level cache: 256-KB, 8-way set associative, 32-byte line size",
[0x83]="2nd-level cache: 512-KB, 8-way set associative, 32-byte line size",
[0x84]="2nd-level cache: 1-MB, 8-way set associative, 32-byte line size",
[0x85]="2nd-level cache: 2-MB, 8-way set associative, 32-byte line size",
[0x86]="2nd-level cache: 512-KB, 4-way set associative, 64-byte line size",
[0x87]="2nd-level cache: 1-MB, 8-way set associative, 64-byte line size",
[0xB0]="Instruction TLB: 4-KB Pages, 4-way set associative, 128 entries",
[0xB1]="UNKNOWN CORE2 DESCRIPTOR 0xB1",
[0xB3]="Data TLB: 4-KB Pages, 4-way set associative, 128 entries",
[0xB4]="Data TLB1: 4-KB Pages, 4-way set associative, 256 entries",
[0xF0]="64-byte Prefetching",
[0xF1]="128-byte Prefetching",
};
#endif

/* AMD volume 3 */
static char *edx8_feat[32] = {
	"fpu","vme","de","pse","tsc","msr","pae","mce",
	"cx8","apic","res10","syscall","mtrr","pge", "mca","cmov",
	"pat","pse36","res18","mp","nx","res21", "mmxext","mmx",
	"fxsr","ffxsr","res26","tscp","res28", "lm","3dnowext","3dnow"
};

static char *ecx8_feat[8] = {
	"ahf64", "cmp", "svm", "res3", "cr8d", "res5", "res6", "res7"
};

static char *edx87_feat[32] = {
	"ts","fid","vid","ttp","tm","stc", "res6", "res7", "fixedtsc",
	"res9", "res10", "res11", "res12", "res13", "res14", "res15",
	"res16", "res17", "res18", "res19", "res20", "res21", "res22",
	"res23", "res24", "res25", "res26", "res27", "res28", "res29",
	"res30", "res31"
};

/* TMTA internal cpuid document */
static char *edx86_feat[32] = {
	"recovery","longrun","pti","lrti","tvm","test","lrdi","sy1",
	"sy2","tsx","tta","topc","pbe","minicms","lrtil","res15",
	"res16","res17","res18","res19","res20","res21","res22","res23",
	"res24","res25","res26","res27","res28","res29","res30","res31"
};

static char *tmta_cpu[8] = {
	"TM100 (P95B)", "TM3200 (P95W)", "TM5400 (P95F)",
	"TM5400 or TM5600 (P95F)", "TM5500 or TM5800 (P95F 0.13)",
	"TM5500 or TM5800 (P95G 0.13)", "TM5700 or TM5900", "TM5700 or TM5900"
};

static char *tmta_package[16] = {
	"unknown", "29mm x 29mm", "21mm x 21mm", "unknown3",
	"unknown4", "unknown5", "unknown6", "unknown7",
	"unknown8", "unknown9", "unknown10", "unknown11",
	"unknown12", "unknown13", "unknown14", "unknown15"
};

static int output_level(uint32_t i, uint32_t *level)
{
	if (memcmp(level, allzero, 16) == 0)
		return 0;		/* nothing to show */

	sprintf(&messagestring[strlen(messagestring)], "0x%08x: "
		"eax=0x%08x ebx=0x%08x ecx=0x%08x edx=0x%08x\n",
		i, level[0], level[1], level[2], level[3]);
	return 1;			/* something to decode */
}

static void output_bits(uint32_t bits, char **names)
{
	int i, col;

	strcat(messagestring, "\t");
	for (i = col = 0; i < 32; i++) {
		if (bits & (1 << i)) {
			col += sprintf(&messagestring[strlen(messagestring)],
					"%s ", names[i]);
		}
		if (col > 60 && i < 31) {
			strcat(messagestring, "\n\t");
			col = 0;
		}
	}
	strcat(messagestring, "\n");
}

#ifdef WIN32
int APIENTRY WinMain(HINSTANCE hCurrentInst, HINSTANCE hPreviousInst,
		     LPSTR lpszCmdLine, int nCmdShow)
#else
int main(int argc, char **argv)
#endif
{
	uint32_t i, f, max, do_tmta, family;


	for (i = 0; i < 32; i++) {
		do_cpuid(tm00info[i], i, 0);
		do_cpuid(tm8kinfo[i], 0x80000000 + i, 0);
		do_cpuid(tm86info[i], 0x80860000 + i, 0);
		do_cpuid(tmc0info[i], 0xc0000000 + i, 0);
		do_cpuid(tmbainfo[i], 0xbabe0000 + i, 0);
	}
	do_cpuid(mystery, 0x8fffffff, 0);
	/* filter out bogus intel replicated levels */
	max = tm00info[0][0] & 0x1f;
	for (i = f = 0; i < 32; i++) {
		if (i > max &&
		    memcmp(tm00info[max], tm00info[i], 16) == 0)
			memset(tm00info[i], 0, 16), f++;
		if (memcmp(tm00info[max], tm8kinfo[i], 16) == 0)
			memset(tm8kinfo[i], 0, 16), f++;
		if (memcmp(tm00info[max], tm86info[i], 16) == 0)
			memset(tm86info[i], 0, 16), f++;
		if (memcmp(tm00info[max], tmc0info[i], 16) == 0)
			memset(tmc0info[i], 0, 16), f++;
		if (memcmp(tm00info[max], tmbainfo[i], 16) == 0)
			memset(tmbainfo[i], 0, 16), f++;
		if (memcmp(tm00info[max], mystery, 16) == 0)
			memset(mystery, 0, 16), f++;
	}

	do_tmta = (tm86info[0][1] == 0x6e617254 && 	/* TransmetaCPU */
	    tm86info[0][2] == 0x55504361 && tm86info[0][3] == 0x74656d73);
	family = (tm00info[1][0] & 0xf00) >> 8;

	if (tm00info[1][3] & (1 << 18)) {
		sprintf(messagestring, "PSN=%08x-%08x-%08x-%08x\n\n",
			tm00info[3][0], tm00info[3][1],
			tm00info[3][2], tm00info[3][3]);
	} else {
		sprintf(messagestring,
			"PSN=DISABLED-DISABLED-DISABLED-DISABLED\n\n");
	}

	if (f) 
		sprintf(&messagestring[strlen(messagestring)],
			"%d levels ignored\n", f);

	/* level 0 */
	output_level(0, tm00info[0]);
	sprintf(&messagestring[strlen(messagestring)],
		"\tMaximum standard function = %d, Vendor ID = ",
		tm00info[0][0]);
	strncat(messagestring, (char *) &tm00info[0][1], 4);
	strncat(messagestring, (char *) &tm00info[0][3], 4);
	strncat(messagestring, (char *) &tm00info[0][2], 4);
	strcat(messagestring, "\n");

	/* level 1 */
	output_level(1, tm00info[1]);
	sprintf(&messagestring[strlen(messagestring)],
		"\tFamily = %d, Model = %d, Stepping = %d",
		family, (tm00info[1][0] & 0xf0) >> 4, (tm00info[1][0] & 0xf));
	if (do_tmta) {
		sprintf(&messagestring[strlen(messagestring)],
			" (%s)", tm00info[1][0] < 0x542 ? "TM100" :
			(family == 5) ?  "Crusoe" : "Efficeon");
	}
	strcat(messagestring, "\n");
	if (family == 15) {
		sprintf(&messagestring[strlen(messagestring)],
			"\tExtended Family = %d, Extended Model = %d\n"
			"\tAPIC ID = %d, Virtual CPUs = %d, "
			"CLFUSH size = %d, Brand ID = 0x%02x\n",
			(tm00info[1][0] >> 20) & 0xff,
			(tm00info[1][0] >> 16) & 0xf,
			(tm00info[1][2] >> 24) & 0xff,
			(tm00info[1][2] >> 16) & 0xff,
			(tm00info[1][2] >> 8) & 0xff,
			tm00info[1][2] & 0xff);
	}
	output_bits(tm00info[1][3], edx1_feat);
	if (tm00info[1][2])	/* ecx feature flags in use? */
		output_bits(tm00info[1][2], ecx1_feat);

	if (output_level(2, tm00info[2])) {
		int byte, c;
		for (i = 0; i < 4; i++) {
		   if (tm00info[2][i] & (1 << 31))
			   continue;	/* not 8-bit descriptors */
		   for (byte = 0; byte < 4; byte++) {
			c = (tm00info[2][i] >> (byte << 3)) & 0xff;
			if (c == 0 || (byte == 0 && i == 0))
				continue;
			sprintf(&messagestring[strlen(messagestring)],
				"\t[%02x] %s\n", c,
#ifdef __GNUC__
				cache_decode[c]
#else
				"msvc lacks c99 [index]=\"value\" array init"
#endif
				);
		   }
		}
	}
	output_level(3, tm00info[3]);
	if (output_level(4, tm00info[4])) {
		sprintf(&messagestring[strlen(messagestring)],
			"\tCache = Level %d %s, "
			"%s%s\n\tthreads sharing this cache = %d\n"
			"\tprocessor cores on this die = %d\n"
			"\tways = %d, line partitions = %d, "
			"line size = %d, sets = %d\n",
			(tm00info[4][0] >> 5) & 7,
			ctype[tm00info[4][0] & 3],
			(tm00info[4][0] & 0x100) ? "Self Initializing, " : "",
			(tm00info[4][0] & 0x200) ? "Fully Associative, " : "",
			1 + ((tm00info[4][0] >> 14) & 0xfff),
			1 + ((tm00info[4][0] >> 26) & 0x3f),
			1 + ((tm00info[4][1] >> 22) & 0x3ff),
			1 + ((tm00info[4][1] >> 12) & 0x3ff),
			1 + (tm00info[4][1] & 0xfff), 1 + tm00info[4][2]);
	}
	if (output_level(5, tm00info[5])) {
		sprintf(&messagestring[strlen(messagestring)],
			"\tSmallest monitor line size = %d bytes,\n"
			"\tLargest monitor line size = %d bytes\n",
			tm00info[5][0] & 65535,
			tm00info[5][1] & 65535);
	}
	if (output_level(6, tm00info[6])) {
		sprintf(&messagestring[strlen(messagestring)],
			"\tDigital Thermometer = %d, "
			"Operating Point Protection = %d\n"
			"\tProgrammable Temp Thresholds = %d\n",
			tm00info[6][0] & 1,
			(tm00info[6][0] >> 2) & 1,
			tm00info[6][1] & 15);
	}
	/* level 7 - 31 */
	for (i = 3; i < 32; i++) {
		output_level(i, tm00info[i]);
	}

	/* level 0x80000000 */
	if (output_level(0x80000000, tm8kinfo[0])) {
		sprintf(&messagestring[strlen(messagestring)],
			"\tMaximum extended function = %d, "
			"Extended Vendor = ",
			tm8kinfo[0][0] & 0x1f);
		strncat(messagestring, (char *) &tm8kinfo[0][1], 4);
		strncat(messagestring, (char *) &tm8kinfo[0][3], 4);
		strncat(messagestring, (char *) &tm8kinfo[0][2], 4);
		if (!tm8kinfo[0][1])	/* ie. Intel Xeon */
			strcat(messagestring, "(null)");
		strcat(messagestring, "\n");
	}
	/* level 0x80000001 */
	if (output_level(0x80000001, tm8kinfo[1])) {
		sprintf(&messagestring[strlen(messagestring)],
			"\tHigh Family = %d, Model = %d, Stepping = %d\n",
			(tm8kinfo[1][0] & 0xf00) >> 8,
			(tm8kinfo[1][0] & 0xf0) >> 4,
			(tm8kinfo[1][0] & 0xf));
		output_bits(tm8kinfo[1][3], edx8_feat);
		if (tm8kinfo[1][2])
			output_bits(tm8kinfo[1][2] & 7, ecx8_feat);
	}
	/* levels 0x80000002 - 4 */
	if (output_level(0x80000002, tm8kinfo[2])) {
		output_level(0x80000003, tm8kinfo[3]);
		output_level(0x80000004, tm8kinfo[4]);
		strcat(messagestring, "\t");
		for (i = 2; i < 5; i++) {
			strncat(messagestring,
				(char *) &tm8kinfo[i], 16);
		}
		strcat(messagestring, "\n");
	}

	/* level 0x80000005 */
	if (output_level(0x80000005, tm8kinfo[5])) {
		if (tm8kinfo[5][0] & 0xffff0000) {
		    sprintf(&messagestring[strlen(messagestring)],
			"\tData TLB 2MB/4MB Pages: %d-way, %d entries\n"
			"\tCode TLB 2MB/4MB Pages: %d-way, %d entries\n",
			(tm8kinfo[5][0] >> 24) & 0xff,
			(tm8kinfo[5][0] >> 16) & 0xff,
			(tm8kinfo[5][0] >> 8) & 0xff,
			tm8kinfo[5][0] & 0xff);
		}
		if (tm8kinfo[5][1] & 0xffff0000) {
		    sprintf(&messagestring[strlen(messagestring)],
			"\tData TLB 4-Kbyte Pages: %d-way, %d entries\n"
			"\tCode TLB 4-Kbyte Pages: %d-way, %d entries\n",
			(tm8kinfo[5][1] >> 24) & 0xff,
			(tm8kinfo[5][1] >> 16) & 0xff,
			(tm8kinfo[5][1] >> 8) & 0xff,
			tm8kinfo[5][1] & 0xff);
		}
		sprintf(&messagestring[strlen(messagestring)],
			"\tL1 Data Cache: %d KB, %d-way, %d bytes per line,"
		        " %d per tag\n"
			"\tL1 Code Cache: %d KB, %d-way, %d bytes per line,"
		        " %d per tag\n",
			(tm8kinfo[5][2] >> 24) & 0xff,
			(tm8kinfo[5][2] >> 16) & 0xff,
			tm8kinfo[5][2] & 0xff,
			(tm8kinfo[5][2] >> 8) & 0xff,
			(tm8kinfo[5][3] >> 24) & 0xff,
			(tm8kinfo[5][3] >> 16) & 0xff,
			tm8kinfo[5][3] & 0xff,
			(tm8kinfo[5][3] >> 8) & 0xff);
	}

	/* level 0x80000006 */
	if (output_level(0x80000006, tm8kinfo[6])) {
		if (tm8kinfo[6][0] & 0xffff0000) {
		    sprintf(&messagestring[strlen(messagestring)],
			"\tL2 Data TLB 2MB/4MB Pages: %s, %d entries\n"
			"\tL2 Code TLB 2MB/4MB Pages: %s, %d entries\n",
			cway[(tm8kinfo[6][0] >> 28) & 0xf],
			(tm8kinfo[6][0] >> 16) & 0xfff,
			cway[(tm8kinfo[6][0] >> 12) & 0xf],
			tm8kinfo[6][0] & 0xfff);
		}
		if (tm8kinfo[6][1] & 0xffff0000) {
		    sprintf(&messagestring[strlen(messagestring)],
			"\tL2 Data TLB 4-Kbyte Pages: %s, %d entries\n"
			"\tL2 Code TLB 4-Kbyte Pages: %s, %d entries\n",
			cway[(tm8kinfo[6][1] >> 28) & 0xf],
			(tm8kinfo[6][1] >> 16) & 0xfff,
			cway[(tm8kinfo[6][1] >> 12) & 0xf],
			tm8kinfo[6][1] & 0xfff);
		}
		sprintf(&messagestring[strlen(messagestring)],
			"\tL2 Data Cache: %d KB, %s, %d bytes per line,"
		        " %d per tag\n",
			(tm8kinfo[6][2] >> 16) & 0xffff,
			cway[(tm8kinfo[6][2] >> 12) & 0xf],
			tm8kinfo[6][2] & 0xff,
			(tm8kinfo[6][2] >> 8) & 0xf);
	}
	/* level 0x80000007 */
	if (output_level(0x80000007, tm8kinfo[7])) {
		output_bits(tm8kinfo[7][3], edx87_feat);
	}

	/* level 0x80000008 */
	if (output_level(0x80000008, tm8kinfo[8])) {
		sprintf(&messagestring[strlen(messagestring)],
			"\tvirtual address size = %d, physical size = %d\n"
			"\tcores per die = %d, maximum cores = %d\n",
			(tm8kinfo[8][0] >> 8) & 0xff,
			tm8kinfo[8][0] & 0xff,
			(tm8kinfo[8][2] & 0xff) + 1,
			(tm8kinfo[8][2] & 0xf000) ?
			1 << ((tm8kinfo[8][2] >> 12) & 15) :
			(tm8kinfo[8][2] & 0xff) + 1);
	}
	output_level(0x80000009, tm8kinfo[9]);
	if (output_level(0x8000000a, tm8kinfo[10])) {
		sprintf(&messagestring[strlen(messagestring)],
			"\tSVM Revision = %d, number of ASIDs = %d\n",
			tm8kinfo[10][0] & 0xff, tm8kinfo[10][1]);
	}
	/* levels 0x8000000b - 31 */
	for (i = 11; i < 32; i++) {
		output_level(i + 0x80000000, tm8kinfo[i]);
	}

	/* level 0x80860000 */
	if (output_level(0x80860000, tm86info[0])) {
		sprintf(&messagestring[strlen(messagestring)],
			"\tMaximum 8086 function = %d, "
			"8086 Vendor = ",
			tm86info[0][0] & 0x1f);
		strncat(messagestring, (char *) &tm86info[0][1], 4);
		strncat(messagestring, (char *) &tm86info[0][3], 4);
		strncat(messagestring, (char *) &tm86info[0][2], 4);
		strcat(messagestring, "\n");
	}
	/* level 0x80860001 */
	if (output_level(0x80860001, tm86info[1])) {
		int procid = tm86info[1][1];
		sprintf(&messagestring[strlen(messagestring)],
			"\t8086 Family = %d, Model = %d, Stepping = %d\n",
			(tm86info[1][0] & 0xf00) >> 8,
			(tm86info[1][0] & 0xf0) >> 4,
			(tm86info[1][0] & 0xf));
		if ((procid >> 16) < 0x200) {
			sprintf(&messagestring[strlen(messagestring)],
				"\t%s %d.%d\n", !procid ? "P95 A0" :
				procid == 0x1000000 ? "P95 B0" :
				(procid & ~0xff) == 0x1000100 ? "P95 B1" :
				tmta_cpu[(procid >> 16) & 0x7],
				(procid >> 8) & 0xff, procid & 0xff);
		}
		sprintf(&messagestring[strlen(messagestring)],
			"\tNominal Frequency = %d MHz\n",
			tm86info[1][2]);
		output_bits(tm86info[1][3], edx86_feat);
	}
	if (output_level(0x80860002, tm86info[2])) {
		if (tm86info[1][1] == 0x02000000) {
			sprintf(&messagestring[strlen(messagestring)],
				"\tEfficeon %d.%d, %s package\n",
				(tm86info[2][0] >> 29) & 7,
				(tm86info[2][0] >> 25) & 15,
				tmta_package[(tm86info[2][0] >> 8) & 15]);
		}
		sprintf(&messagestring[strlen(messagestring)],
			"\tCMS Revision %d.%d.%d-%d-%d\n",
			(tm86info[2][1] >> 24) & 0xff,
			(tm86info[2][1] >> 16) & 0xff,
			(tm86info[2][1] >> 8) & 0xff,
			tm86info[2][1] & 0xff, tm86info[2][2]);
	}
	/* levels 0x80860003 - 6 */
	if (output_level(0x80860003, tm86info[3])) {
		output_level(0x80860004, tm86info[4]);
		output_level(0x80860005, tm86info[5]);
		output_level(0x80860006, tm86info[6]);
		strcat(messagestring, "\t");
		for (i = 3; i < 7; i++) {
			strncat(messagestring,
				(char *) &tm86info[i], 16);
		}
		strcat(messagestring, "\n");
	}
	/* level 0x80860007 */
	if (output_level(0x80860007, tm86info[7])) {
		sprintf(&messagestring[strlen(messagestring)],
			"\tCurrent Frequency = %d MHz @ %dmV, %d%%\n",
			tm86info[7][0], tm86info[7][1], tm86info[7][2]);
	}
	/* levels 0x80860008 - 32 */
	for (i = 8; i < 32; i++) {
		output_level(i + 0x80860000, tm86info[i]);
	}
	for (i = 0; i < 32; i++) {
		output_level(i + 0xc0000000, tmc0info[i]);
	}
	for (i = 0; i < 32; i++) {
		output_level(i + 0xbabe0000, tmbainfo[i]);
	}
	if (output_level(0x8fffffff, mystery)) {
		strcat(messagestring, "\t");
		strncat(messagestring, (char *) &mystery, 16);
		strcat(messagestring, "\n");
	}
#ifdef WIN32
	MessageBox(NULL, messagestring, "idtool 1.5 - AMD/Intel/Transmeta CPUID VIEWER",
		   MB_OK | MB_ICONINFORMATION);
#else
	fprintf(stderr, "idtool 1.5 - AMD/Intel/Transmeta CPUID VIEWER\n\n%s\n",
			messagestring);
#endif
	return 0;
}

