/* fbtv.c 
 *   by Nathan Laredo (laredo@gnu.org)
 *   See file ChangeLog for recent changes.
 */

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <unistd.h>
#include <string.h>
#include <ctype.h>
#include <fcntl.h>
#include <errno.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/time.h>
#include <linux/types.h>
#include <linux/videodev.h>
#include "channels.h"

/* For TV card driving */
struct video_window vwindow;
struct video_window vsavew;
struct video_tuner vtuner;
struct video_picture vpicture;
struct video_buffer vbuffer;
struct video_capability vcap;
struct video_channel vinput;
struct video_audio vaudio;

int tvfd;
int format = NTSC_BROADCAST;
int channel = 10;
int do_scan = 0;
int do_gray;
int freqtune = 0;
int capture_running = 0;
int audio_running = 1;
int channel_mute = 0;

char tmpbuf[256];	/* temp string space for abuse */

static char *formatname[11] =
{
	"NTSC BROADCAST ", "NTSC CABLE     ", "NTSC JP BRDCAST", "NTSC JP CABLE  ",
	"PAL EUROPE     ", "PAL ITALY      ", "PAL NEW ZEALAND", "PAL AUSTRALIA  ",
	"PAL UHF/IRELAND", "PAL CABLE(B)(G)", "SECAM          "
};

/* the following variables are set by get_framebuffer()  */
int *framebuf, bytes_per_line, bits_per_pixel, framewidth, frameheight;

void setformatfreq(int freq)
{
	int i = format;
	ioctl(tvfd, VIDIOCGTUNER, &vtuner);
	if (format <= NTSC_JP_BCAST) {
		vtuner.flags = VIDEO_TUNER_NTSC;
		vtuner.mode = VIDEO_MODE_NTSC;
	} else {
		if (format == 10) {		/* SECAM */
			vtuner.mode = VIDEO_MODE_SECAM;
			vtuner.flags = VIDEO_TUNER_SECAM;
			/* SECAM tuning is wrong, as I saw on my visit to Paris */
			i = PAL_UHF_GHI;	/* fixme!  I have no secam charts! */
		} else {
			vtuner.mode = VIDEO_MODE_PAL;
			vtuner.tuner = VIDEO_TUNER_PAL;
		}
	}
	ioctl(tvfd, VIDIOCSTUNER, &vtuner);
	if (!(vcap.type & VID_TYPE_TUNER))
		return;					/* can't tune */
	freq = tvtuner[freq].freq[i];
	if (!freq)					/* no need to set invalid freq */
		return;
	freq *= 16;
	freq /= 1000;
	freqtune = freq;
	ioctl(tvfd, VIDIOCSFREQ, &freq);
}

void channel_up(void)
{
	int i = format;
	if (format == 10)
		i = PAL_UHF_GHI;
	if ((++channel) == CHAN_ENTRIES)
		channel = 0;
	while (tvtuner[channel].freq[i] == 0) {
		channel++;
		if (channel == CHAN_ENTRIES)
			channel = 0;
	}
	setformatfreq(channel);
}

void channel_down(void)
{
	int i = format;
	if (format == 10)
		i = PAL_UHF_GHI;
	if ((--channel) < 0)
		channel = CHAN_ENTRIES - 1;
	while (tvtuner[channel].freq[i] == 0) {
		channel--;
		if (channel < 0)
			channel = CHAN_ENTRIES - 1;
	}
	setformatfreq(channel);
}


void do_audio(int n)
{
	if (ioctl(tvfd, VIDIOCGAUDIO, &vaudio) == 0) {
		if (n)
			vaudio.flags &= ~VIDEO_AUDIO_MUTE;
		else
			vaudio.flags |= VIDEO_AUDIO_MUTE;
		if (ioctl(tvfd, VIDIOCSAUDIO, &vaudio) < 0)
			perror("set audio");
	}
}

int setup_tv_device(int n)
{
	sprintf(tmpbuf, "/dev/video%d", n);
	if ((tvfd = open(tmpbuf, O_RDWR)) < 0) {
		perror("open");
		return -1;
	}
	if (ioctl(tvfd, VIDIOCGCAP, &vcap) < 0) {
		perror("get capabilities");
		return -1;
	}
	if (vcap.audios > 0)
		do_audio(1);

	vwindow.clips = NULL;
	vwindow.clipcount = 0;
	vwindow.width = 640;
	vwindow.height = 480;
	vwindow.x = 0; 
	vwindow.y = 0;
	ioctl(tvfd, VIDIOCGPICT, &vpicture);
	vpicture.depth = bits_per_pixel;
	switch (bits_per_pixel) {
	case 8:
		vpicture.palette = VIDEO_PALETTE_HI240;
		break;
	case 15:
		vpicture.palette = VIDEO_PALETTE_RAW;
		break;
	case 16:
		vpicture.palette = VIDEO_PALETTE_RGB565;
		break;
	case 24:
		vpicture.palette = VIDEO_PALETTE_RGB24;
		break;
	case 32:
		vpicture.palette = VIDEO_PALETTE_RGB32;
		break;
	default:
		fprintf(stderr, "unsupported depth %d.\n",
				bits_per_pixel);
		return -1;
	}
	vpicture.contrast = 32768;
	if (ioctl(tvfd, VIDIOCSPICT, &vpicture) < 0) {
		perror("set depth");
		fprintf(stderr, "Unable to set %d bit palette\n",
				bits_per_pixel);
		return -1;
	}
	/* these values are set by earlier call to get_framebuffer() */
	vbuffer.width = framewidth;
	vbuffer.height = frameheight;
	vbuffer.base = framebuf;
	if ((vbuffer.depth = bits_per_pixel) == 15)
		vbuffer.depth++;
	vbuffer.bytesperline = bytes_per_line;
	if (ioctl(tvfd, VIDIOCSFBUF, &vbuffer) < 0) {
		perror("set buffer");
		return -1;
	}
	ioctl(tvfd, VIDIOCSWIN, &vwindow);

	return 0;
}

void tvset_on() { int i = 1; ioctl(tvfd, VIDIOCCAPTURE, &i); }

void close_tv_card()
{
	int i = 0;
	ioctl(tvfd, VIDIOCCAPTURE, &i);
	close(tvfd);
}

int input_quiet = 0;
void set_input(int n)
{
	if (ioctl(tvfd, VIDIOCGCHAN, &vinput) < 0)	/* get name */
		perror("get channel");
	vinput.channel = n;
	vinput.norm = VIDEO_MODE_NTSC;
	if (ioctl(tvfd, VIDIOCSCHAN, &vinput) < 0)
		perror("set channel");
	if (ioctl(tvfd, VIDIOCGCHAN, &vinput) < 0)	/* get name */
		perror("get channel");
	if (n == 0)					/* tuner */
		setformatfreq(channel);
}

/* the default colour table, for VGA+ colour systems */
int default_red[] =
{0x00, 0xaa, 0x00, 0xaa, 0x00, 0xaa, 0x00, 0xaa,
 0x55, 0xff, 0x55, 0xff, 0x55, 0xff, 0x55, 0xff};
int default_grn[] =
{0x00, 0x00, 0xaa, 0x55, 0x00, 0x00, 0xaa, 0xaa,
 0x55, 0x55, 0xff, 0xff, 0x55, 0x55, 0xff, 0xff};
int default_blu[] =
{0x00, 0x00, 0x00, 0x00, 0xaa, 0xaa, 0xaa, 0xaa,
 0x55, 0x55, 0x55, 0x55, 0xff, 0xff, 0xff, 0xff};

#if 0
/* Setup pseudocolor color cube or grayscale */
void set_colors(void)
{
	int i, j;
	if ((cmap = XCreateColormap(mydisplay, mywindow, attribs.visual,
								AllocAll)) == 0) {	/* allocate all colors */
		if (!do_gray)
			fprintf(stderr, "unable to allocate colors, using existing map\n");
		return;
	}
	for (i = 0; i < 256; i++) {
		mycolor[i].flags = DoRed | DoGreen | DoBlue;
		mycolor[i].pixel = i;	/* initialize for following Query */
	}
	/* get existing map to preserve some entries if necessary */
	XQueryColors(mydisplay, cmap, mycolor, 256);

	for (i = 0; i < 256; i++)
		if (i < 16 || i > 240 || do_gray) {
			if (do_gray)
				mycolor[i].red = mycolor[i].green = mycolor[i].blue = i * 255;
			else {
				if (i > 239)
					j = i - 240;
				else
					j = i;
				mycolor[i].red = default_red[j] * 255;
				mycolor[i].green = default_grn[j] * 255;
				mycolor[i].blue = default_blu[j] * 255;
			}
		} else {
			j = i - 16;
			if (j > 224)
				break;
			else {
				mycolor[i].red = ((j % 25) / 5) * 16383;
				mycolor[i].green = (j / 25) * 8191;
				mycolor[i].blue = (j % 5) * 16383;
			}
		}
	XStoreColors(mydisplay, cmap, mycolor, 255);
	XSetWindowColormap(mydisplay, mywindow, cmap);
}
#endif

/* this routine fills framebuf, bytes_per_line, bits_per_pixel, and do_gray */
/* if 8bpp depth is detected, set_colors() is called */
int get_framebuffer(void)
{
	framewidth = 1024;
	frameheight = 4000;
	bits_per_pixel = 16;
	do_gray = 0;
	framebuf = 0xce000000;
	bytes_per_line = framewidth;
	if (bits_per_pixel == 15)
		bytes_per_line *= 2;
	else
		bytes_per_line *= (bits_per_pixel / 8);
#if 0
	if (bits_per_pixel == 8)
		set_colors();
#endif
	return 0;
}

/* for keyboad entry of channels, c = 0 or 3 chars will enter */
/* On-scren-display of channel as it is entered is done here */

void ch_add_char(char c)
{
	static char cn[5];
	int i, found = -1;

	if (cn[4] != ' ') {			/* initialization flag */
		cn[4] = cn[0] = cn[1] = cn[2] = ' ';
		cn[3] = 0;
	}
	/* rotate new char into channel selector string */
	if (c) {
		cn[0] = cn[1];
		cn[1] = cn[2];
		cn[2] = c;
		cn[3] = 0;
	}
	if (c == 0 || cn[0] != ' ' || cn[1] > '2') {	/* find & change */
		for (i = 0; i < CHAN_ENTRIES && found < 0; i++)
			if (!strcmp(cn, tvtuner[i].name))
				found = i;
		if (found >= 0)
			setformatfreq(channel = found);
		cn[0] = cn[1] = cn[2] = ' ';
		return;
	}
}

int main(int argc, char **argv)
{
	int done = 0, retval = 0, tvdevice = 0;
	fd_set rfds;
	struct timeval tv;

	if (get_framebuffer() < 0)
		return 1;
	if (argc > 1)
		tvdevice = atoi(argv[1]);
	if (setup_tv_device(tvdevice) < 0)
		return 2;
	set_input(1);
	tvset_on();
	/* Main event loop */
	while (!done) {
		if (getchar()=='q')
			done++;
	}	/* while !done */
	close_tv_card();
	exit(0);
}
