/* io_X11.c 
 *   by Nathan Laredo (laredo@gnu.org)
 *
 *   See file ChangeLog for recent changes.
 */

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/keysym.h>
#include <X11/extensions/xf86dga.h>
#include <X11/extensions/xf86vmode.h>
#include <X11/Intrinsic.h>

unsigned long myforeground, mybackground;
XSetWindowAttributes myattribs;	/* for window creation */
XWindowAttributes attribs;	/* for vis, depth, width, height */
XFontStruct *font_struct;
XColor mycolor[256];		/* used for PseudoColor and logo mapping */
Display *mydisplay;
XSizeHints myhint;
Window mywindow;
Cursor mycursor;
XColor curs_col;
Pixmap curs_pix;
char title[256];		/* window title */
XEvent myevent;
long myevents;			/* keep track of our events */
Colormap cmap;
KeySym mykey;
int myscreen;
GC mygc;

/* VideoMode swapping support for full screen tvset */
int full_screen = 0;
XF86VidModeModeInfo **modelines, *fullmode = NULL, *vidmode = NULL;

void vidmode_find(void)
{
    int k, num, clock;
    XF86VidModeModeLine mline;

    /* Save the current video mode so we can get back here */
    XF86VidModeGetModeLine(mydisplay, 0, &clock, &mline);

    /* Find us a nice 640x480 video mode :) */
    XF86VidModeGetAllModeLines(mydisplay, 0, &num, &modelines);
    for (k = 0; k < num; k++) {
	if ((modelines[k]->hdisplay == 640) && (modelines[k]->vdisplay == 480))
	    fullmode = modelines[k];
	if ((modelines[k]->hdisplay == mline.hdisplay) &&
	    (modelines[k]->vdisplay == mline.vdisplay))
	    vidmode = modelines[k];
    }

    {
#if 0
	Window junk;
	XTranslateCoordinates(mydisplay, mywindow,
			      DefaultRootWindow(mydisplay), 0, 0,
			      &initial_x, &initial_y, &junk);
#endif
    }

    if ((fullmode == NULL) || (vidmode == NULL))
	printf("error finding a 640x480 modeline, no fullscreen support\n");
}

void vidmode_640(void)
{
    int tmp = capture_running;
    do_capture(0);
    XF86VidModeSwitchToMode(mydisplay, 0, fullmode);
    if (tmp)
	do_capture(1);
}

void vidmode_def(void)
{
    int tmp = capture_running;
    do_capture(0);
    XF86VidModeSwitchToMode(mydisplay, 0, vidmode);
    if (tmp)
	do_capture(1);
}

void grab_fullscreen(void)
{
    do_capture(0);
    vsavew = vwindow;
    XMoveResizeWindow(mydisplay, mywindow, 0, 0, 640, 480);
    vwindow.x = 0;
    vwindow.y = 0;
    vwindow.width = 640;
    vwindow.height = 480;
    vidmode_640();
    XGrabPointer(mydisplay, mywindow, True, 0, GrabModeAsync, GrabModeAsync,
		 mywindow, None, CurrentTime);
    XDefineCursor(mydisplay, mywindow, mycursor);
    XF86VidModeSetViewPort(mydisplay, 0, vwindow.x, vwindow.y);
    XRaiseWindow(mydisplay, mywindow);
    do_capture(1);
    full_screen = 1;
}

void release_fullscreen(void)
{
    do_capture(0);
    XUngrabPointer(mydisplay, CurrentTime);
    XUndefineCursor(mydisplay, mywindow);
    XMoveResizeWindow(mydisplay, mywindow, vsavew.x, vsavew.y,
		      vsavew.width, vsavew.height);
    vidmode_def();
    vwindow = vsavew;
    do_capture(1);
    full_screen = 0;
}

/* end full screen support code */

/* Setup pseudocolor color cube or grayscale */
void set_colors(void)
{
    int i, j;
    if ((cmap = XCreateColormap(mydisplay, mywindow, attribs.visual,
				AllocAll)) == 0) {	/* allocate all colors */
	if (!do_gray)
	    fprintf(stderr, "unable to allocate colors, using existing map\n");
	return;
    }
    for (i = 0; i < 256; i++) {
	mycolor[i].flags = DoRed | DoGreen | DoBlue;
	mycolor[i].pixel = i;	/* initialize for following Query */
    }
    /* get existing map to preserve some entries if necessary */
    XQueryColors(mydisplay, cmap, mycolor, 256);

    for (i = 0; i < 256; i++)
	if (i < 16 || do_gray) {
	    if (do_gray)
		mycolor[i].red = mycolor[i].green = mycolor[i].blue = i * 255;
	    /* preserve first 16 colors on non-grayscale */
	} else {
	    j = i - 16;
	    if (j > 224)
		break;
	    else {
		mycolor[i].red = ((j % 25) / 5) * 16383;
		mycolor[i].green = (j / 25) * 8191;
		mycolor[i].blue = (j % 5) * 16383;
	    }
	}
    XStoreColors(mydisplay, cmap, mycolor, 255);
    XSetWindowColormap(mydisplay, mywindow, cmap);
}

/* this routine fills framebuf, bytes_per_line, bits_per_pixel, and do_gray */
/* if 8bpp depth is detected, set_colors() is called */
int get_framebuffer(void)
{
    int major, minor, bank, ram;	/* values we throw away */

    framewidth = attribs.width;
    frameheight = attribs.height;
    if (attribs.depth == 24)
	bits_per_pixel = (attribs.visual->map_entries) / 8;
    else
	bits_per_pixel = attribs.depth;
    if (bits_per_pixel < 8) {
	fprintf(stderr, "%d bpp is not supported\n",
		bits_per_pixel);
	return -1;
    }
    if (attribs.visual->class == GrayScale ||
	attribs.visual->class == StaticGray)
	do_gray = 1;
    else
	do_gray = 0;

    if (XF86DGAQueryVersion(mydisplay, &major, &minor))
	XF86DGAGetVideoLL(mydisplay, myscreen, (int *) &framebuf,
			  &bytes_per_line, &bank, &ram);
    else {
	fprintf(stderr, "XF86DGA not available.\n");
	bytes_per_line=1920;
	bits_per_pixel=16;
	framebuf = 0xe8000000;
    }

    if (bits_per_pixel == 15)
	bytes_per_line *= 2;
    else
	bytes_per_line *= (bits_per_pixel / 8);
    if (bits_per_pixel == 8)
	set_colors();
    fprintf(stderr,"framewidth=%d, frameheight=%d, base=%08x, bpp=%d, bpl=%d\n",
	    framewidth, frameheight, framebuf, bits_per_pixel, bytes_per_line);
    return 0;
}

void update_title(void)
{
    char finebuf[8];

    if (finetune)
	sprintf(finebuf, "%+03d", finetune);
    else
	strcpy(finebuf, "");
    switch (titlestyle) {
    case 0:
	sprintf(title, "tvset - %s %s %s%s", formatname[format],
		vinput.name, tvtuner[channel].name, finebuf);
	break;
    default:
	sprintf(title, "tvset");
    }
    XStoreName(mydisplay, mywindow, title);
    XSetIconName(mydisplay, mywindow, title);
}

/* for keyboad entry of channels, c = 0 or 3 chars will enter */
/* On-scren-display of channel as it is entered is done here */

void ch_add_char(char c)
{
    static char cn[5];
    int i, found = -1;

    if (cn[4] != ' ') {		/* initialization flag */
	cn[4] = cn[0] = cn[1] = cn[2] = ' ';
	cn[3] = 0;
    }
    /* rotate new char into channel selector string */
    if (c) {
	cn[0] = cn[1];
	cn[1] = cn[2];
	cn[2] = c;
	cn[3] = 0;
    }
    if (c == 0 || cn[0] != ' ' || cn[1] >= '3') {	/* find & change */
	for (i = 0; i < CHAN_ENTRIES && found < 0; i++)
	    if (!strcmp(cn, tvtuner[i].name))
		found = i;
	if (found >= 0)
	    setformatfreq(channel = found);
	cn[0] = cn[1] = cn[2] = ' ';
	update_title();
	return;
    }
}

int handle_event(void)
{
    Window junk;		/* Scratch Window */

    switch (myevent.type) {
    case VisibilityNotify:
	switch (myevent.xvisibility.state) {
	case VisibilityUnobscured:
	    break;
	case VisibilityFullyObscured:
	    break;
	case VisibilityPartiallyObscured:
	    break;
	}
	break;
    case ConfigureNotify:
	XTranslateCoordinates(mydisplay, mywindow,
			      DefaultRootWindow(mydisplay), 0, 0,
			      &vwindow.x, &vwindow.y, &junk);
	vwindow.width = myevent.xconfigure.width;
	vwindow.height = myevent.xconfigure.height;
	do_capture(1);
	break;
    case MapNotify:
	do_capture(1);
	break;
    case UnmapNotify:
	do_capture(0);
	break;
    case DestroyNotify:
	if (myevent.xdestroywindow.window != mywindow)
	    break;
	do_capture(0);
	return 1;		/* DONE */
	break;
    case Expose:
	do_capture(1);
	break;
    case MappingNotify:
	XRefreshKeyboardMapping((XMappingEvent *) & myevent);
	break;
    case KeyPress:
	/* all commands implemented via keypress currently */
	mykey = XKeycodeToKeysym(mydisplay, myevent.xkey.keycode, 0);
	switch (mykey) {
	case XK_q:
	    return 1;		/* DONE */
	    break;
	case XK_Escape:
	    if (full_screen)
		release_fullscreen();
	    else
		grab_fullscreen();
	    break;
	case XK_KP_Divide:
	    if ((vwindow.width * 3) / (vwindow.height * 4) >= 1)
		vwindow.width = (vwindow.height * 4) / 3;
	    else
		vwindow.height = (vwindow.width * 3) / 4;
	    XResizeWindow(mydisplay, mywindow,
			  vwindow.width, vwindow.height);
	    XTranslateCoordinates(mydisplay, mywindow,
				  DefaultRootWindow(mydisplay), 0, 0,
				  &vwindow.x, &vwindow.y, &junk);
	    do_capture(1);
	    break;
	case XK_KP_Multiply:
	    XResizeWindow(mydisplay, mywindow, 160, 120);
	    XTranslateCoordinates(mydisplay, mywindow,
				  DefaultRootWindow(mydisplay), 0, 0,
				  &vwindow.x, &vwindow.y, &junk);
	    vwindow.width = 160;
	    vwindow.height = 120;
	    do_capture(1);
	    break;
	case XK_KP_Subtract:
	    XResizeWindow(mydisplay, mywindow, 320, 240);
	    XTranslateCoordinates(mydisplay, mywindow,
				  DefaultRootWindow(mydisplay), 0, 0,
				  &vwindow.x, &vwindow.y, &junk);
	    vwindow.width = 320;
	    vwindow.height = 240;
	    do_capture(1);
	    break;
	case XK_KP_Add:
	    XResizeWindow(mydisplay, mywindow, 640, 480);
	    XTranslateCoordinates(mydisplay, mywindow,
				  DefaultRootWindow(mydisplay), 0, 0,
				  &vwindow.x, &vwindow.y, &junk);
	    vwindow.width = 640;
	    vwindow.height = 480;
	    do_capture(1);
	    break;
	case XK_KP_Decimal:
	case XK_m:
	    audio_running = 1 - audio_running;
	    do_audio(audio_running);
	    break;
	case XK_t:
	    if (++titlestyle > 1)
		titlestyle = 0;
	    update_title();
	    break;
	case XK_f:
	    if (++format > 10)
		format = 0;
	    setformatfreq(channel);
	    update_title();
	    break;
	case XK_Next:		/* Page Up */
	    update_title();
	    break;
	case XK_Prior:		/* Page Dn */
	    update_title();
	    break;
	case XK_Up:
	    channel_up();
	    update_title();
	    break;
	case XK_Down:
	    channel_down();
	    update_title();
	    break;
	case XK_i:
	    if (++vinput.channel >= vcap.channels)
		vinput.channel = 0;
	    set_input(vinput.channel);
	    update_title();
	    break;
	case XK_1:
	case XK_KP_1:
	    ch_add_char('1');
	    break;
	case XK_2:
	case XK_KP_2:
	    ch_add_char('2');
	    break;
	case XK_3:
	case XK_KP_3:
	    ch_add_char('3');
	    break;
	case XK_4:
	case XK_KP_4:
	    ch_add_char('4');
	    break;
	case XK_5:
	case XK_KP_5:
	    ch_add_char('5');
	    break;
	case XK_6:
	case XK_KP_6:
	    ch_add_char('6');
	    break;
	case XK_7:
	case XK_KP_7:
	    ch_add_char('7');
	    break;
	case XK_8:
	case XK_KP_8:
	    ch_add_char('8');
	    break;
	case XK_9:
	case XK_KP_9:
	    ch_add_char('9');
	    break;
	case XK_0:
	case XK_KP_0:
	    ch_add_char('0');
	    break;
	case XK_a:
	    ch_add_char('A');
	    break;
	case XK_e:
	    ch_add_char('E');
	    break;
	case XK_s:
	    ch_add_char('S');
	    break;
	case XK_ISO_Enter:
	case XK_KP_Enter:
	    ch_add_char(0);
	    break;
	case XK_Left:
	    finetune--;
	    if (finetune < -100) {
		finetune = 100;
		channel_down();
	    }
	    setformatfreq(channel);
	    update_title();
	    break;
	case XK_Right:
	    finetune++;
	    if (finetune > 100) {
		finetune = -100;
		channel_up();
	    }
	    setformatfreq(channel);
	    update_title();
	    break;
	case XK_BackSpace:
	    break;
	}
	break;
    default:
	/* Ignore all other events */
    }				/* switch */
    return 0;
}

int init_tvset(int argc, char **argv)
{
    XGCValues gcvals;
    Window junk;
    int done;

    if (!getenv("DISPLAY")) {
	fprintf(stderr, "DISPLAY not set.\n");
	exit(1);
    }
    if (!(mydisplay = XOpenDisplay(getenv("DISPLAY")))) {
	fprintf(stderr, "Unable to open display.\n");
	exit(1);
    }
    *(vinput.name) = 0;

    XSynchronize(mydisplay, 1);

    XGetWindowAttributes(mydisplay, DefaultRootWindow(mydisplay), &attribs);

    myscreen = DefaultScreen(mydisplay);
    if (attribs.depth != 8) {
	mybackground = BlackPixel(mydisplay, myscreen);
	myforeground = WhitePixel(mydisplay, myscreen);
    } else {			/* pseudocolor / grayscale pixel defs */
	mybackground = 22;
	myforeground = 192;	/* text will be drawn this color */
    }
    /* this sets up reasonable min/max sizes as well as a 1.3-1.5 aspect */
    myhint.width = 320;
    myhint.height = 240;
    myhint.max_width = 640;
    myhint.max_height = 480;
    myhint.min_width = 180;
    myhint.min_width = 120;
    myhint.min_aspect.x = 4;
    myhint.min_aspect.y = 3;
    myhint.max_aspect.x = 15;
    myhint.max_aspect.y = 10;
    /* to add aspect ratio restriction, add to mask below */
    myhint.flags = PSize | PMinSize | PMaxSize /*| PAspect */ ;

    myevents = myattribs.event_mask =
	KeyPressMask |
	KeyReleaseMask |
	ButtonPressMask |
	ButtonReleaseMask |
	ButtonMotionMask |
	ExposureMask |
	VisibilityChangeMask |
	StructureNotifyMask |
	SubstructureNotifyMask;

    myattribs.background_pixel = mybackground;
    mywindow = XCreateWindow(mydisplay, DefaultRootWindow(mydisplay),
	0, 0, myhint.width, myhint.height, 0, attribs.depth, InputOutput,
		  attribs.visual, CWEventMask | CWBackPixel, &myattribs);
    XSetStandardProperties(mydisplay, mywindow, title, title, None, argv,
			   argc, &myhint);
    gcvals.foreground = myforeground;
    gcvals.background = mybackground;
    mygc = XCreateGC(mydisplay, mywindow, GCForeground | GCBackground, &gcvals);
    XMapRaised(mydisplay, mywindow);

    /* NOTE: get_framebuffer must succeed before tv card init */
    done = get_framebuffer();	/* fall through to end on failure */
    if (!done)
	done = setup_tv_device(0);	/* fall through on failure */

    XTranslateCoordinates(mydisplay, mywindow,
			  DefaultRootWindow(mydisplay), 0, 0,
			  &vwindow.x, &vwindow.y, &junk);
    vwindow.width = 320;
    vwindow.height = 240;

    font_struct = XLoadQueryFont(mydisplay, "12x24");
    XSetFont(mydisplay, mygc, font_struct->fid);

    /* Scan X VideoModes for full screen support */
#if 0
    vidmode_find();
#endif

    /* Create a 1 pixel cursor to use in full screen mode */
    curs_pix = XCreatePixmap(mydisplay, mywindow, 1, 1, 1);
    curs_col.pixel = 0;
    curs_col.red = 0;
    curs_col.green = 0;
    curs_col.blue = 0;
    curs_col.flags = 0;
    curs_col.pad = 0;
    mycursor = XCreatePixmapCursor(mydisplay, curs_pix, curs_pix, &curs_col,
				   &curs_col, 1, 1);

    return done;
}

void refreshTV(Display * disp)
{
    XSetWindowAttributes xswa;
    unsigned long mask;
    static Window rwin;

    xswa.event_mask = 0;
    XChangeWindowAttributes(mydisplay, mywindow, CWEventMask, &xswa);

    if (!rwin) {
	xswa.override_redirect = True;
	xswa.backing_store = NotUseful;
	xswa.save_under = False;
	xswa.background_pixel = BlackPixel(mydisplay, myscreen);

	mask = (CWBackPixel | CWSaveUnder | CWBackingStore |
		CWOverrideRedirect);

	rwin = XCreateWindow(disp, RootWindow(mydisplay, myscreen),
			     0, 0, vbuffer.width, vbuffer.height, 0,
			     CopyFromParent, InputOutput, CopyFromParent,
			     mask, &xswa);
    }
    XMapRaised(mydisplay, rwin);
    XUnmapWindow(mydisplay, rwin);

    xswa.event_mask = myevents;
    XChangeWindowAttributes(mydisplay, mywindow, CWEventMask, &xswa);
}

int main(int argc, char **argv)
{
    int done = 0, retval = 0, xfd;
    fd_set rfds;
    struct timeval tv;

    done = init_tvset(argc, argv);
    update_title();

    do_capture(1);		/* Fire up the tube... */
    xfd = ConnectionNumber(mydisplay);

    /* Main event loop */
    while (!done) {
	tv.tv_sec = 1;	/* all timeouts are 1 second multiples */
	tv.tv_usec = 0;
	FD_ZERO(&rfds);
	FD_SET(xfd, &rfds);
	retval = select(FD_SETSIZE, &rfds, NULL, NULL, &tv);
	if (FD_ISSET(xfd, &rfds))
	    while (XPending(mydisplay)) {
		XNextEvent(mydisplay, &myevent);
		done = handle_event();
	    }
	if (!retval) {
	    /* osd expire */
	}
    }				/* while */
    if (full_screen)
	release_fullscreen();
    close_tv_card();
    XFreeGC(mydisplay, mygc);
    XDestroyWindow(mydisplay, mywindow);
    XCloseDisplay(mydisplay);
    exit(0);
}				/* main */

