/* iotest.c 
 *   by Nathan Laredo (laredo@gnu.org)
 *
 * ChangeLog:
 * 22 Feb 01 Initial version
 */

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <unistd.h>
#include <string.h>
#include <ctype.h>
#include <fcntl.h>
#include <errno.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/time.h>
#include <linux/types.h>

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/keysym.h>
#include <X11/extensions/xf86dga.h>
#include <X11/Intrinsic.h>

Display *mydisplay;
XWindowAttributes attribs;
int myscreen;
volatile int *framebuf;
int bytes_per_line, bits_per_pixel, framewidth, frameheight;

void init_display(void)
{
	int major, minor, bank, ram;	/* values we throw away */

	if (!getenv("DISPLAY")) {
		fprintf(stderr, "DISPLAY not set.\n");
		exit(1);
	}
	if (!(mydisplay = XOpenDisplay(getenv("DISPLAY")))) {
		fprintf(stderr, "Unable to open display.\n");
		exit(1);
	}
	myscreen = DefaultScreen(mydisplay);

	XSynchronize(mydisplay, True);

	XGetWindowAttributes(mydisplay, DefaultRootWindow(mydisplay),
			     &attribs);

	if (attribs.depth == 24)
		bits_per_pixel = (attribs.visual->map_entries) / 8;
	else
		bits_per_pixel = attribs.depth;
	if (bits_per_pixel < 8) {
		fprintf(stderr, "%d bpp is not supported\n",
			bits_per_pixel);
		exit(1);
	}
	if (XF86DGAQueryVersion(mydisplay, &major, &minor)) {
		fprintf(stderr, "DGA Version %d.%d detected\n", major, minor);
		XF86DGAGetVideo(mydisplay, myscreen, &framebuf,
				&bytes_per_line, &bank, &ram);
	} else {
		fprintf(stderr, "XF86DGA not available.\n");
		exit(1);
	}
	if (bits_per_pixel == 15)
		bytes_per_line *= 2;
	else
		bytes_per_line *= (bits_per_pixel / 8);

	framewidth = attribs.width;
	frameheight = attribs.height;
}

int main(int argc, char **argv)
{
	struct timeval tv_start, tv_stop;
	int i;

	init_display();
	fprintf(stderr, "Detected a %dx%dx%d display.  %d bytes per line.\n",
			framewidth, frameheight, bits_per_pixel,
			bytes_per_line);

	fprintf(stderr, "\nTesting sequential burst writes to videomem.\n");
	gettimeofday(&tv_start, NULL);

	for (i = 0; i < 100; i++) {
		memset(framebuf, i & 255, bytes_per_line * frameheight);
	}

	gettimeofday(&tv_stop, NULL);

	fprintf(stderr, "wrote %d bytes sequentially to display.\n",
			bytes_per_line * frameheight * i);
	fprintf(stderr, "stop time  = %10d.%06d\n", (int)tv_stop.tv_sec,
			(int)tv_stop.tv_usec);
	fprintf(stderr, "start time = %10d.%06d\n", (int)tv_start.tv_sec,
			(int)tv_start.tv_usec);
	tv_stop.tv_usec -= tv_start.tv_usec;
	tv_stop.tv_sec -= tv_start.tv_sec;
	if (tv_stop.tv_usec < 0) {
		tv_stop.tv_usec += 1000000;
		tv_stop.tv_sec--;
	}
	fprintf(stderr, "elapsed    = %10d.%06d\t", (int)tv_stop.tv_sec,
			(int)tv_stop.tv_usec);

	fprintf(stderr, "%16.6f bytes/sec\n\n",
		(float)(bytes_per_line * frameheight * i) /
		((float)tv_stop.tv_sec + (float)tv_stop.tv_usec / 1000000.0));

	fprintf(stderr, "Testing repeated writes to the same video memory.\n");

	gettimeofday(&tv_start, NULL);

	for (i = 0; i < bytes_per_line * frameheight * 100; i++) {
		(volatile) framebuf[0] = i;
	}

	gettimeofday(&tv_stop, NULL);

	fprintf(stderr, "wrote %d 32-bit writes to the same video memory.\n",
			i);
	fprintf(stderr, "stop time  = %10d.%06d\n", (int)tv_stop.tv_sec,
			(int)tv_stop.tv_usec);
	fprintf(stderr, "start time = %10d.%06d\n", (int)tv_start.tv_sec,
			(int)tv_start.tv_usec);
	tv_stop.tv_usec -= tv_start.tv_usec;
	tv_stop.tv_sec -= tv_start.tv_sec;
	if (tv_stop.tv_usec < 0) {
		tv_stop.tv_usec += 1000000;
		tv_stop.tv_sec--;
	}
	fprintf(stderr, "elapsed    = %10d.%06d\t", (int)tv_stop.tv_sec,
			(int)tv_stop.tv_usec);

	fprintf(stderr, "%16.6f bytes/sec\n", (4.0 * (float)i) /
		((float)tv_stop.tv_sec + (float)tv_stop.tv_usec / 1000000.0));

	exit(0);
}
