/* xaudio.c 
 *   by Nathan Laredo (laredo@gnu.org)
 */

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <unistd.h>
#include <string.h>
#include <ctype.h>
#include <fcntl.h>
#include <errno.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/soundcard.h>
#include <math.h>

#include <fftw.h>
#include <rfftw.h>

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/keysym.h>
#include <X11/Intrinsic.h>

Display *mydisplay;
Window mywindow;
Pixmap mypixmap;
GC mygcR, mygcG, mygcB, mygcbg;
XGCValues gcvals;
XEvent myevent;
int myscreen;
XSizeHints myhint;
KeySym mykey;
XColor mycolor;
XFontStruct *font_struct;
unsigned long myforeground, mybackground;
XWindowAttributes attribs;	/* for vis, depth, width, height */
XSetWindowAttributes myattribs;	/* for window creation */
XPoint aupoints[1024];		/* for drawing data */
long myevents;			/* keep track of our events */
char title[4096];		/* window title */
unsigned short aubuf[2048];	/* audio sample buf */
float shbuf[2048];		/* superheterodyne buf */

fftw_real in[2048], out[2048], hwin[2048];
rfftw_plan p;

char *notes[12] = {
	"C ", "C#", "D ", "D#",
	"E ", "F ", "F#", "G ",
	"G#", "A ", "A#", "B "
};

unsigned char freqtable[65536];	/* holds note number for 0-65536Hz */
int midi_pitch[128];		/* holds frequency for midi note number */

void init_freqtable(void)
{
	int i, j, delta, mindelta, minnote = 0;
	double p;

	for (i = 0; i < 128; i++) {
		p = (double)13.75 *
			pow((double)2, (double)(i - 9) / (double)12.0);
		midi_pitch[i] = (int)p;
	}
	for (i = 0; i < 65536; i++) {
		/* find closest midi note for frequency and record in table */
		mindelta = abs(midi_pitch[minnote] - i);
		for (j = minnote; j < 128; j++) {
			delta = abs(midi_pitch[j] - i);
			if (delta < mindelta) {
				minnote = j;
				mindelta = delta;
			}
			if (delta > mindelta)
				break;
		}
		freqtable[i] = minnote;
	}
}
	

void init_display(int argc, char **argv)
{
	int i;

	p = rfftw_create_plan(1024, FFTW_REAL_TO_COMPLEX, FFTW_MEASURE);
	for (i = 0; i < 2048; i++) {
		hwin[i] = sin(i * (M_PI / 2048.0));
	}

	if (!getenv("DISPLAY")) {
		fprintf(stderr, "DISPLAY not set.\n");
		exit(1);
	}
	if (!(mydisplay = XOpenDisplay(getenv("DISPLAY")))) {
		fprintf(stderr, "Unable to open display.\n");
		exit(1);
	}
	mybackground = BlackPixel(mydisplay, myscreen);
	myforeground = WhitePixel(mydisplay, myscreen);
	myhint.width = 1024;
	myhint.height = 256;
	myhint.max_width = 1024;
	myhint.max_height = 256;
	myhint.min_width = 1024;
	myhint.min_width = 256;
	myhint.x = 0;
	myhint.y = 0;
	/* to add aspect ratio restriction, add to mask below */
	myhint.flags = PSize | PMinSize | PMaxSize | PPosition;
	myevents = myattribs.event_mask = KeyPressMask | ButtonPressMask;
	myattribs.background_pixel = mybackground;
	mywindow = XCreateWindow(mydisplay, DefaultRootWindow(mydisplay),
				 myhint.x, myhint.y, myhint.width,
				 myhint.height, 0, attribs.depth, InputOutput,
				 attribs.visual, CWEventMask | CWBackPixel,
				 &myattribs);
	sprintf(title, "xaudio debugger");
	XSetStandardProperties(mydisplay, mywindow, title, title, None,
			       argv, argc, &myhint);
	gcvals.foreground = mybackground;
	gcvals.background = myforeground;
	mygcbg = XCreateGC(mydisplay, mywindow,
			   GCForeground | GCBackground, &gcvals);
	mycolor.flags = DoRed | DoGreen | DoBlue;
	mycolor.red = 65535;
	mycolor.green = 0;
	mycolor.blue = 0;
	mycolor.pad = 0;
	XAllocColor(mydisplay, DefaultColormap(mydisplay, myscreen),
		    &mycolor);
	gcvals.foreground = mycolor.pixel;
	gcvals.background = mybackground;
	gcvals.function = GXor;
	mygcR = XCreateGC(mydisplay, mywindow,
			  GCForeground | GCBackground | GCFunction, &gcvals);
	mycolor.flags = DoRed | DoGreen | DoBlue;
	mycolor.red = 0;
	mycolor.green = 65535;
	mycolor.blue = 0;
	mycolor.pad = 0;
	XAllocColor(mydisplay, DefaultColormap(mydisplay, myscreen),
		    &mycolor);
	gcvals.foreground = mycolor.pixel;
	mygcG = XCreateGC(mydisplay, mywindow,
			  GCForeground | GCBackground | GCFunction, &gcvals);
	mycolor.red = 0;
	mycolor.green = 0;
	mycolor.blue = 65535;
	mycolor.pad = 0;
	XAllocColor(mydisplay, DefaultColormap(mydisplay, myscreen),
		    &mycolor);
	gcvals.foreground = mycolor.pixel;
	mygcB = XCreateGC(mydisplay, mywindow,
			  GCForeground | GCBackground | GCFunction, &gcvals);
	XMapRaised(mydisplay, mywindow);
	mypixmap = XCreatePixmap(mydisplay, mywindow, 1024, 256,
				 DefaultDepth(mydisplay, myscreen));
	XFillRectangle(mydisplay, mypixmap, mygcbg, 0, 0, 1024, 256);
	XSetWindowBackgroundPixmap(mydisplay, mywindow, mypixmap);
	font_struct = XLoadQueryFont(mydisplay, "12x24");
	XSetFont(mydisplay, mygcG, font_struct->fid);
}

int main(int argc, char **argv)
{
	fd_set rdfs;
	int aufd, xfd, done = 0, i, align = 0;
	int samplerate, delta, max, val;
	float fbase, shfreq = 0.0, shlen;

	if (argc > 1) {
		shfreq = atof(argv[1]);
	}
	init_freqtable();
	if ((aufd = open("/dev/dsp", O_RDONLY)) < 0) {
		perror("open /dev/dsp");
		exit(1);
	}
	samplerate = 48000;
	ioctl(aufd, SNDCTL_DSP_SPEED, &samplerate);
	fprintf(stderr, "samplerate=%d\n", samplerate);
	i = 16;
	ioctl(aufd, SNDCTL_DSP_SETFMT, &i);
	fprintf(stderr, "setfmt=%d\n", i);
	fbase = (float)samplerate / 2048.0;

	shlen = 2.0 * M_PI * shfreq / (float)samplerate;
	for (i = 0; i < 2048; i++) {
		/* use cosine: if shfreq = 0, shbuf[i] = 1.0 */
		shbuf[i] = cos(i * shlen);
	}
	shlen = 0.0;
	if (shfreq > 0.0) {
		shlen = 2.0 * (float)samplerate / shfreq;
	}

	init_display(argc, argv);

	for (i = 0; i < 1024; i++)
		aupoints[i].x = i;

	xfd = ConnectionNumber(mydisplay);
	/* Main event loop */
	while (!done) {
		FD_ZERO(&rdfs);
		FD_SET(xfd, &rdfs);
		FD_SET(aufd, &rdfs);
#if 0
		select(FD_SETSIZE, &rdfs, NULL, NULL, NULL);
		if (FD_ISSET(xfd, &rdfs))
#endif
			XFlush(mydisplay);
			while (XPending(mydisplay)) {
				XNextEvent(mydisplay, &myevent);
				if (myevent.type == ButtonPress) {
					sprintf(title, "xaudio %d %d",
						myevent.xbutton.x,
						256 - myevent.xbutton.y);
					XStoreName(mydisplay, mywindow,
						   title);
				}
			}
#if 0
		if (FD_ISSET(aufd, &rdfs)) {
#endif
			if (read(aufd, aubuf, 2048) < 2048)
				done++;
			for (i = 0; i < 1024; i++) {
				aubuf[i] = (aubuf[i] >> 8) + 128;
			}

			/* slice and dice */
			if (shlen > 1.0 && shlen < 2047.0) {
				int win = (int)shlen;
				int a, j;

				for (i = 0; i < win; i++) {
					a = 0;
					for (j = 0; j < 2048 / win; j++) {
						a += aubuf[(i + j) % 2048];
					}
					aubuf[i] = (unsigned char) (a / j);
				}
			}

			/* superheterodyne tuning */
			for (i = 0; i < 2048; i++) {
				aubuf[i] = (unsigned char) (((float)aubuf[i]
					- 128.0) * shbuf[i] + 128.0);
			}

			/* simple average filter */
			if (shlen > 1.0) {
				int win = (int)shlen;
				int a, j;

				for (i = win; i < 2048 - win; i++) {
					a = 0;
					for (j = -win; j < win; j++) {
						a += aubuf[i + j];
					}
					aubuf[i] = (unsigned char) (a /
						(win * 2));
				}
			}

			for (i = 0; i < 2048; i++) {
				/* window and scale the data 1/257 * s * w */
				in[i] = 0.00389105 * aubuf[i]; // * hwin[i];
			}
			rfftw_one(p, &in[align], out);

			XFillRectangle(mydisplay, mypixmap, mygcbg, 0, 0,
				       1024, 256);
			/* convert freq output to coordinates and find max */
			max = 0;
			delta = 0;
			for (i = 0; i < 12; i++)
				aupoints[i].y = 255;
			for (i = 0; i < 512; i++) {
				val = out[i];
				aupoints[i].y = 255 - val;
				if (val > max && i > 12) {
					max = val;
					delta = i;
				}
			}

			/* draw real in blue */
			XDrawLines(mydisplay, mypixmap, mygcB, aupoints,
				   512, CoordModeOrigin);

			for (i = 0; i < 512; i++) {
				val = out[1024 - i];
				aupoints[i].y = 255 - val;
				if (val > max && i > 12) {
					max = val;
					delta = i;
				}
			}

			/* draw imaginary in red */
			XDrawLines(mydisplay, mypixmap, mygcR, aupoints,
				   512, CoordModeOrigin);

			i = (int)((float)delta * fbase);
			max = freqtable[i];

			if (i > 20) {
				sprintf(title, "%4dHz %4s%2d midi[0x%02x]"
					"%+5dHz", i, notes[max % 12],
					max / 12, max, i - midi_pitch[max]);
				delta = samplerate / i; 
			}

			XDrawString(mydisplay, mypixmap, mygcG, 24, 24,
				    title, 32);

			for (i = 0; i < 1024; i++)
				aupoints[i].y = 256 - aubuf[i + align];

			/* draw samples in green */
			XDrawLines(mydisplay, mypixmap, mygcG, aupoints,
				   1024, CoordModeOrigin);

			XClearWindow(mydisplay, mywindow);	/* repaint */
#if 0
		}
#endif
	}			/* while */
	rfftw_destroy_plan(p);
	close(aufd);
	XFreeGC(mydisplay, mygcR);
	XFreeGC(mydisplay, mygcG);
	XFreeGC(mydisplay, mygcB);
	XFreeGC(mydisplay, mygcbg);
	XFreePixmap(mydisplay, mypixmap);
	XDestroyWindow(mydisplay, mywindow);
	XCloseDisplay(mydisplay);
	exit(0);
}				/* main */
